import os

def process_directory(directory):
    """
    Process all `.fil` files in the given directory and rename them in alphabetical order.
    Recursively processes subdirectories.
    """
    # Get all `.fil` files and sort them alphabetically
    fil_files = [f for f in os.listdir(directory) if f.lower().endswith('.fil')]
    fil_files.sort()

    # Rename files in the current directory
    for counter, filename in enumerate(fil_files):
        # Construct the new filename
        shortname = get_shortname(filename)
        
        if counter > 99:
            shortname = shortname[:5]
        
        new_name = f"{counter:02d}{shortname}.FIL"

        # Rename the file
        old_path = os.path.join(directory, filename)
        new_path = os.path.join(directory, new_name)

        print(f"Renaming '{old_path}' to '{new_path}'")
        os.rename(old_path, new_path)

    # Process subdirectories recursively
    for entry in os.listdir(directory):
        subdirectory = os.path.join(directory, entry)
        if os.path.isdir(subdirectory):
            process_directory(subdirectory)

def get_shortname(filename):
    """
    Extract the first six letters from the filename, ignoring numbers and spaces.
    If fewer than six letters are available, pad with "DKSONG".
    """
    # Remove the extension
    name_without_extension = os.path.splitext(filename)[0]

    # Extract letters only and convert to uppercase
    letters = ''.join(c.upper() for c in name_without_extension if c.isalpha())

    # Ensure the shortname is exactly 6 characters long
    if len(letters) < 6:
        letters += "DKSONG"

    return letters[:6]

if __name__ == "__main__":
    import argparse

    # Parse command-line arguments
    parser = argparse.ArgumentParser(description="Rename .fil files in a directory recursively.")
    parser.add_argument("directory", help="The root directory to process.")
    args = parser.parse_args()

    # Ensure the provided path is a directory
    if not os.path.isdir(args.directory):
        print(f"Error: {args.directory} is not a valid directory.")
        exit(1)

    # Start processing
    print(f"Processing directory: {args.directory}")
    process_directory(args.directory)
    print("Renaming completed.")
